//
// Assets and source code of SCP Security Systems are licensed under Creative Commons Attribution-ShareAlike 3.0
//
import { CommandResult, Entity, EntityQueryOptions, Player, Vector3, world } from "@minecraft/server";
import { ActionFormData } from "@minecraft/server-ui";
import { OVERWORLD, getHostPlayer } from "./main";
import { getClosestEntityFromLocation, getClosestPlayerFromLocation } from "./utils";


const EQO_KEYPAD: EntityQueryOptions = {
  families: ["keypad"]
}

const KEYPAD_FORM_DIGIT_1 = new ActionFormData()
  .title("§c§lSECURITY DOOR SYSTEM")
  .body("Enter Code: §lDIGIT 1")
  .button("0")
  .button("1")
  .button("2")
  .button("3")
  .button("4")
  .button("5")
  .button("6")
  .button("7")
  .button("8")
  .button("9")
  .button("Cancel");

const KEYPAD_FORM_DIGIT_2 = new ActionFormData()
  .title("§c§lSECURITY DOOR SYSTEM")
  .body("Enter Code: §lDIGIT 2")
  .button("0")
  .button("1")
  .button("2")
  .button("3")
  .button("4")
  .button("5")
  .button("6")
  .button("7")
  .button("8")
  .button("9");

const KEYPAD_FORM_DIGIT_3 = new ActionFormData()
  .title("§c§lSECURITY DOOR SYSTEM")
  .body("Enter Code: §lDIGIT 3")
  .button("0")
  .button("1")
  .button("2")
  .button("3")
  .button("4")
  .button("5")
  .button("6")
  .button("7")
  .button("8")
  .button("9")
  .button("Cancel");

const KEYPAD_FORM_DIGIT_4 = new ActionFormData()
  .title("§c§lSECURITY DOOR SYSTEM")
  .body("Enter Code: §lDIGIT 4")
  .button("0")
  .button("1")
  .button("2")
  .button("3")
  .button("4")
  .button("5")
  .button("6")
  .button("7")
  .button("8")
  .button("9")
  .button("Cancel");


/**
 * Get form of given index.
 * Returns KEYPAD_FORM_DIGIT_1 if given index is invalid.
 * @param index 
 * @returns KEYPAD_FORM_DIGIT_(index)
 */
function getFormOfIndex(index: number): ActionFormData {
  let form: ActionFormData;
  switch (index) {
    case 0:
      form = KEYPAD_FORM_DIGIT_1
      break;
    case 1:
      form = KEYPAD_FORM_DIGIT_2
      break;
    case 2:
      form = KEYPAD_FORM_DIGIT_3
      break;
    case 3:
      form = KEYPAD_FORM_DIGIT_4
      break;
    default:
      form = KEYPAD_FORM_DIGIT_1;
      break;
  }
  return form;
}


function cancelPasswordEnteringMode(keypadDigitUiHelperEntity: Entity, keypadEntity: Entity) {
  keypadEntity.triggerEvent('keypad:cancel_password_entering_mode');
  keypadDigitUiHelperEntity.triggerEvent('despawn');
}


function confirmEnteredPassword(keypadDigitUiHelperEntity: Entity, keypadEntity: Entity) {
  keypadEntity.triggerEvent('keypad:confirm_entered_password');
  keypadDigitUiHelperEntity.triggerEvent('despawn');
}


function startKeypadDigitForm(keypadDigitUiHelperEntity: Entity, keypadEntity: Entity, playerToSeeForm: Player, index: number) {
  if (keypadDigitUiHelperEntity === undefined) {
    world.sendMessage('§ckeypad.js: keypadDigitUiHelperEntity is undefined!')
    return;
  }

  if (!keypadDigitUiHelperEntity.isValid()) {
    world.sendMessage('§ekeypad.js: keypadDigitUiHelperEntity is invalid. This might cause an error.');
  }
  
  if (keypadEntity === undefined) {
    world.sendMessage('§ckeypad.js: keypadEntity is undefined!')
    return;
  }

  if (!keypadEntity.isValid()) {
    world.sendMessage('§ekeypad.js: keypadEntity is invalid. This might cause an error.');
  }

  if (playerToSeeForm === undefined) {
    world.sendMessage('§ckeypad.js: playerToSeeForm is undefined!')
    return;
  }

  if (!playerToSeeForm.isValid()) {
    world.sendMessage('§ekeypad.js: playerToSeeForm is invalid. This might cause an error.');
  }

  if (index < 0) {
    world.sendMessage('§ckeypad.js: Invalid index!');
    return;
  }
  
  // If this function is called when index arg is 4, it means player has entered all the 4 digits.
  if (index >= 4) {
    confirmEnteredPassword(keypadDigitUiHelperEntity, keypadEntity)
    return;
  }

  let form: ActionFormData = getFormOfIndex(index);

  form.show(playerToSeeForm).then(response => {
    let selectionIndex: number = response.selection === undefined ? 10 : response.selection; // selectionIndex also means selected number

    let isCanceled: boolean = selectionIndex >= 10 || response.canceled;
    if (isCanceled) {
      cancelPasswordEnteringMode(keypadDigitUiHelperEntity, keypadEntity)
      return;
    }

    let keypadMode: number = keypadEntity.getProperty('keypad:mode') as number;
    let currentDigitIndex: number = keypadEntity.getProperty('keypad:current_digit_index') as number;
    
    // Admin setup mode
    if (keypadMode === 0) {

      if (currentDigitIndex === 0) {
        keypadEntity.setProperty('keypad:admin_digit_1', selectionIndex);
      }

      else if (currentDigitIndex === 1) {
        keypadEntity.setProperty('keypad:admin_digit_2', selectionIndex);
      }

      else if (currentDigitIndex === 2) {
        keypadEntity.setProperty('keypad:admin_digit_3', selectionIndex);
      }

      else if (currentDigitIndex === 3) {
        keypadEntity.setProperty('keypad:admin_digit_4', selectionIndex);
      }

      else {
        world.sendMessage('§cERROR');
        cancelPasswordEnteringMode(keypadDigitUiHelperEntity, keypadEntity)
      }
    }

    // Normal password entering mode
    else if (keypadMode === 2) {

      if (currentDigitIndex === 0) {
        keypadEntity.setProperty('keypad:digit_1', selectionIndex);
      }

      else if (currentDigitIndex === 1) {
        keypadEntity.setProperty('keypad:digit_2', selectionIndex);
      }

      else if (currentDigitIndex === 2) {
        keypadEntity.setProperty('keypad:digit_3', selectionIndex);
      }

      else if (currentDigitIndex === 3) {
        keypadEntity.setProperty('keypad:digit_4', selectionIndex);
      }

      else {
        world.sendMessage('§cERROR');
        cancelPasswordEnteringMode(keypadDigitUiHelperEntity, keypadEntity)
      }
    }

    else {
      world.sendMessage('§cERROR');
      cancelPasswordEnteringMode(keypadDigitUiHelperEntity, keypadEntity)
    }

    keypadEntity.triggerEvent('keypad:increment_current_digit_index');

    index++;
    startKeypadDigitForm(keypadDigitUiHelperEntity, keypadEntity, playerToSeeForm, index);
  });
}


export function setupKeypadSystem() {
  world.afterEvents.entitySpawn.subscribe(event => {
    let spawnedEntity = event.entity;
    if (spawnedEntity.typeId != 'lc:scpd_keypad_digit_ui_helper') return;

    // The entity lc:scpd_keypad_digit_ui_helper is used to get keypad entity

    let keypadEntity = getClosestEntityFromLocation(EQO_KEYPAD, OVERWORLD, spawnedEntity.location) as Entity;
    if (keypadEntity === undefined) {
      world.sendMessage('§cCannot start keypad form operation if there is no keypad entity near lc:scpd_keypad_digit_ui_helper!');
      spawnedEntity.triggerEvent('despawn');
      return;
    }

    let overlapCheckResult: CommandResult = spawnedEntity.runCommand('testfor @e[type=lc:scpd_keypad,r=1]');
    if (overlapCheckResult.successCount > 1) {
      world.sendMessage('§cToo many keypad entities in close range!');
      spawnedEntity?.runCommand('event entity @e[type=lc:scpd_keypad,r=0.7] despawn');
      spawnedEntity?.triggerEvent('despawn');
      return;
    }

    let playerToSeeForm = getClosestPlayerFromLocation(getHostPlayer(), spawnedEntity.location);

    startKeypadDigitForm(spawnedEntity, keypadEntity, playerToSeeForm, 0);
  });
}